//OHSAT GAMES TUTORIAL: MEGALAGA 7 BONUS: Enemy Bullets
//There's a reference to Rolling in the OHSAT Tutorial for this lesson.
//It is your duty to make the reference to Weird Al and not Fred Durst or something equally terrible.   

//https://www.ohsat.com/tutorial/#mega-drive-tutorials 

#include <genesis.h>
#include <resources.h>

#define LEFT_EDGE 0
#define RIGHT_EDGE 320
#define BOTTOM_EDGE 224

#define MAX_ENEMIES 6
#define MAX_FLY_ENEMIES 3
#define MAX_BULLETS 3
#define MAX_ENEMY_BULLETS 5
#define FLY_SPAWN_DELAY 180 // approx 3 seconds @ 60 FPS

#define ANIM_STRAIGHT 0
#define ANIM_MOVE 1
#define ANIM_DIE 7

typedef struct {
    int x, y, w, h;
    int velx, vely;
    int health;
    Sprite* sprite;
    char name[8];
} Entity;

// Globals
int offset = 0;
u16 enemiesLeft = 0;
int score = 0;
char hud_string[40];
bool flyEnemiesSpawned = FALSE;
u16 flySpawnTimer = 0;

Entity player = {0}, player_2 = {0};
Entity enemies_top[MAX_ENEMIES];
Entity enemies_bottom[MAX_FLY_ENEMIES];
Entity bullets_p1[MAX_BULLETS], bullets_p2[MAX_BULLETS];
Entity enemyBulletsTop[MAX_ENEMY_BULLETS], enemyBulletsFly[MAX_ENEMY_BULLETS];

u16 bulletsOnScreenP1 = 0, bulletsOnScreenP2 = 0;
u16 enemyBulletsOnScreenTop = 0, enemyBulletsOnScreenFly = 0;

u16 enemyFireCooldownTop = 0;
u16 enemyFireCooldownFly = 0;

// Function declarations
void killEntity(Entity* e);
void reviveEntity(Entity* e);
void positionPlayers();
void positionEnemies();
void positionBullets();
void positionEnemyBullets();
void shootBullet(Entity* shooter, Entity* bulletArray, u16* bulletCount, int velY);
void enemyFire();
void handleCollisions();
void updateScoreDisplay();
void spawnFlyEnemies();
void myJoyHandler(u16 joy, u16 changed, u16 state);

int main() {
    JOY_init();
    JOY_setEventHandler(&myJoyHandler);

    SYS_disableInts();
    VDP_loadTileSet(background.tileset, 1, DMA);
    PAL_setPalette(PAL1, background.palette->data, DMA);
    PAL_setPalette(PAL2, background.palette->data, DMA);
    VDP_setScrollingMode(HSCROLL_PLANE, VSCROLL_PLANE);
    PAL_setColor(34, RGB24_TO_VDPCOLOR(0x0078f8));
    SYS_enableInts();

    SPR_init();

    // Init players
    player = (Entity){152, 192, 16, 16, 0, 0, 1, SPR_addSprite(&ship, 152, 192, TILE_ATTR(PAL1, 0, FALSE, FALSE)), "P1"};
    player_2 = (Entity){100, 192, 16, 16, 0, 0, 1, SPR_addSprite(&ship1, 100, 192, TILE_ATTR(PAL1, 0, FALSE, FALSE)), "P2"};

    // Fill background with stars
    for (int i = 0; i < 1280; i++) {
        int thex = i % 40;
        int they = i / 40;
        int val = (random() % 10) + 1;
        if (val > 3) val = 1;
        VDP_setTileMapXY(BG_B, TILE_ATTR_FULL(PAL1, 0, 0, 0, val), thex, they);
    }

    // Initialize top enemies
    enemiesLeft = 0;
    for (int i = 0; i < MAX_ENEMIES; i++) {
        Sprite* spriteTop = SPR_addSprite(&ship, i * 48, 32, TILE_ATTR(PAL2, 0, TRUE, FALSE));
        if (spriteTop) {
            enemies_top[i] = (Entity){i * 48, 32, 16, 16, 1, 0, 1, spriteTop, ""};
            sprintf(enemies_top[i].name, "E1_%d", i);
            enemiesLeft++;
        } else {
            enemies_top[i].health = 0;
            enemies_top[i].sprite = NULL;
            VDP_drawText("Top sprite failed", 2, 12 + i);
        }
    }

    // Initialize fly enemies inactive (no sprite yet)
    for (int i = 0; i < MAX_FLY_ENEMIES; i++) {
        enemies_bottom[i].health = 0;
        enemies_bottom[i].sprite = NULL;
    }

    // Initialize bullets (off-screen, hidden)
    for (int i = 0; i < MAX_BULLETS; i++) {
        bullets_p1[i] = (Entity){0, -10, 8, 8, 0, 0, 0, SPR_addSprite(&bullet, 0, -10, TILE_ATTR(PAL1, 0, FALSE, FALSE)), ""};
        bullets_p2[i] = (Entity){0, -10, 8, 8, 0, 0, 0, SPR_addSprite(&bullet1, 0, -10, TILE_ATTR(PAL1, 0, FALSE, FALSE)), ""};
        killEntity(&bullets_p1[i]);
        killEntity(&bullets_p2[i]);
    }

    // Initialize enemy bullets (top enemies, bullet sprite)
    for (int i = 0; i < MAX_ENEMY_BULLETS; i++) {
        enemyBulletsTop[i] = (Entity){0, 0, 8, 8, 0, 0, 0, SPR_addSprite(&bullet, 0, -10, TILE_ATTR(PAL2, 0, FALSE, FALSE)), ""};
        killEntity(&enemyBulletsTop[i]);
    }

    // Initialize enemy bullets (fly enemies, bullet1 sprite)
    for (int i = 0; i < MAX_ENEMY_BULLETS; i++) {
        enemyBulletsFly[i] = (Entity){0, 0, 8, 8, 0, 0, 0, SPR_addSprite(&bullet1, 0, -10, TILE_ATTR(PAL2, 0, FALSE, FALSE)), ""};
        killEntity(&enemyBulletsFly[i]);
    }

    updateScoreDisplay();

    while (1) {
        VDP_setVerticalScroll(BG_B, offset -= 2);
        if (offset <= -256) offset = 0;

        positionPlayers();
        positionEnemies();
        positionBullets();
        positionEnemyBullets();
        enemyFire();
        handleCollisions();

        SPR_update();
        SYS_doVBlankProcess();
    }

    return 0;
}

// Input Handler
void myJoyHandler(u16 joy, u16 changed, u16 state) {
    Entity* p = (joy == JOY_1) ? &player : &player_2;
    Entity* bullets = (joy == JOY_1) ? bullets_p1 : bullets_p2;
    u16* bulletCount = (joy == JOY_1) ? &bulletsOnScreenP1 : &bulletsOnScreenP2;

    if (state & BUTTON_RIGHT) {
        p->velx = 2;
        SPR_setAnim(p->sprite, ANIM_MOVE);
        SPR_setHFlip(p->sprite, TRUE);
    } else if (state & BUTTON_LEFT) {
        p->velx = -2;
        SPR_setAnim(p->sprite, ANIM_MOVE);
        SPR_setHFlip(p->sprite, FALSE);
    } else if ((changed & (BUTTON_LEFT | BUTTON_RIGHT))) {
        p->velx = 0;
        SPR_setAnim(p->sprite, ANIM_STRAIGHT);
    }

    if ((state & BUTTON_B) && (changed & BUTTON_B)) {
        shootBullet(p, bullets, bulletCount, -3);
    }
}

// Movement and position updates
void positionPlayers() {
    Entity* players[2] = {&player, &player_2};
    for (int i = 0; i < 2; i++) {
        Entity* p = players[i];
        p->x += p->velx;

        if (p->x < LEFT_EDGE) p->x = LEFT_EDGE;
        if (p->x + p->w > RIGHT_EDGE) p->x = RIGHT_EDGE - p->w;

        SPR_setPosition(p->sprite, p->x, p->y);
    }
}

void positionEnemies() {
    // Top enemies
    for (int i = 0; i < MAX_ENEMIES; i++) {
        Entity* e = &enemies_top[i];
        if (e->health > 0) {
            e->x += e->velx;
            if (e->x < LEFT_EDGE || (e->x + e->w) > RIGHT_EDGE) e->velx = -e->velx;
            SPR_setPosition(e->sprite, e->x, e->y);
        }
    }
    // Flying enemies
    for (int i = 0; i < MAX_FLY_ENEMIES; i++) {
        Entity* f = &enemies_bottom[i];
        if (f->health > 0) {
            f->x += f->velx;

            if (f->x < LEFT_EDGE) {
                f->x = LEFT_EDGE;
                f->velx = -f->velx;
            } else if (f->x + f->w > RIGHT_EDGE) {
                f->x = RIGHT_EDGE - f->w;
                f->velx = -f->velx;
            }
            SPR_setPosition(f->sprite, f->x, f->y);
        }
    }
}

void shootBullet(Entity* shooter, Entity* bulletArray, u16* bulletCount, int velY) {
    if (*bulletCount < MAX_BULLETS) {
        for (int i = 0; i < MAX_BULLETS; i++) {
            Entity* b = &bulletArray[i];
            if (b->health == 0) {
                b->x = shooter->x + (shooter->w / 2) - (b->w / 2);
                b->y = shooter->y + (velY < 0 ? -b->h : shooter->h); // spawn above if going up, below if going down
                b->vely = velY;
                reviveEntity(b);
                SPR_setPosition(b->sprite, b->x, b->y);
                (*bulletCount)++;
                break;
            }
        }
    }
}

void positionBullets() {
    for (int i = 0; i < MAX_BULLETS; i++) {
        // Player 1 bullets
        Entity* b1 = &bullets_p1[i];
        if (b1->health > 0) {
            b1->y += b1->vely;
            if (b1->y + b1->h < 0) {
                killEntity(b1);
                if (bulletsOnScreenP1 > 0) bulletsOnScreenP1--;
            } else {
                SPR_setPosition(b1->sprite, b1->x, b1->y);
            }
        }

        // Player 2 bullets
        Entity* b2 = &bullets_p2[i];
        if (b2->health > 0) {
            b2->y += b2->vely;
            if (b2->y + b2->h < 0) {
                killEntity(b2);
                if (bulletsOnScreenP2 > 0) bulletsOnScreenP2--;
            } else {
                SPR_setPosition(b2->sprite, b2->x, b2->y);
            }
        }
    }
}

void positionEnemyBullets() {
    // Enemy bullets from top enemies
    for (int i = 0; i < MAX_ENEMY_BULLETS; i++) {
        Entity* eb = &enemyBulletsTop[i];
        if (eb->health > 0) {
            eb->y += eb->vely;
            if (eb->y > BOTTOM_EDGE) {
                killEntity(eb);
                if (enemyBulletsOnScreenTop > 0) enemyBulletsOnScreenTop--;
            } else {
                SPR_setVFlip(eb->sprite, TRUE); // Ensure vertical flip
                SPR_setPosition(eb->sprite, eb->x, eb->y);
            }
        }
    }

    // Enemy bullets from fly enemies
    for (int i = 0; i < MAX_ENEMY_BULLETS; i++) {
        Entity* eb = &enemyBulletsFly[i];
        if (eb->health > 0) {
            eb->y += eb->vely;
            if (eb->y > BOTTOM_EDGE) {
                killEntity(eb);
                if (enemyBulletsOnScreenFly > 0) enemyBulletsOnScreenFly--;
            } else {
                SPR_setVFlip(eb->sprite, TRUE); // Ensure vertical flip
                SPR_setPosition(eb->sprite, eb->x, eb->y);
            }
        }
    }
}

void enemyFire() {
    // Cooldown timers decrement
    if (enemyFireCooldownTop > 0) enemyFireCooldownTop--;
    if (enemyFireCooldownFly > 0) enemyFireCooldownFly--;

    // Fire from top enemies
    if (enemyFireCooldownTop == 0 && enemiesLeft > 0) {
        for (int i = 0; i < MAX_ENEMIES; i++) {
            Entity* e = &enemies_top[i];
            if (e->health > 0 && enemyBulletsOnScreenTop < MAX_ENEMY_BULLETS) {
                if (random() % 20 == 0) { // 1/20 chance to fire
                    for (int j = 0; j < MAX_ENEMY_BULLETS; j++) {
                        Entity* b = &enemyBulletsTop[j];
                        if (b->health == 0) {
                            b->x = e->x + (e->w / 2) - (b->w / 2);
                            b->y = e->y + e->h;
                            b->vely = 2;
                            reviveEntity(b);
                            enemyBulletsOnScreenTop++;
                            enemyFireCooldownTop = 30;
                            break;
                        }
                    }
                    break;
                }
            }
        }
    }

    // Spawn fly enemies after top enemies all die
    if (enemiesLeft == 0) {
        flySpawnTimer++;
        if (!flyEnemiesSpawned && flySpawnTimer > FLY_SPAWN_DELAY) {
            spawnFlyEnemies();
            flyEnemiesSpawned = TRUE;
        }
    }

    // Fire from fly enemies
    if (flyEnemiesSpawned && enemyFireCooldownFly == 0) {
        for (int i = 0; i < MAX_FLY_ENEMIES; i++) {
            Entity* f = &enemies_bottom[i];
            if (f->health > 0 && enemyBulletsOnScreenFly < MAX_ENEMY_BULLETS) {
                if (random() % 20 == 0) {
                    for (int j = 0; j < MAX_ENEMY_BULLETS; j++) {
                        Entity* b = &enemyBulletsFly[j];
                        if (b->health == 0) {
                            b->x = f->x + (f->w / 2) - (b->w / 2);
                            b->y = f->y + f->h;
                            b->vely = 2;
                            reviveEntity(b);
                            enemyBulletsOnScreenFly++;
                            enemyFireCooldownFly = 30;
                            break;
                        }
                    }
                    break;
                }
            }
        }
    }
}

void handleCollisions() {
    // Player bullets vs top enemies
    for (int i = 0; i < MAX_BULLETS; i++) {
        Entity* b1 = &bullets_p1[i];
        if (b1->health > 0) {
            for (int j = 0; j < MAX_ENEMIES; j++) {
                Entity* e = &enemies_top[j];
                if (e->health > 0) {
                    if (b1->x < e->x + e->w && b1->x + b1->w > e->x &&
                        b1->y < e->y + e->h && b1->y + b1->h > e->y) {
                        killEntity(b1);
                        bulletsOnScreenP1--;
                        e->health--;
                        if (e->health <= 0) {
                            killEntity(e);
                            enemiesLeft--;
                            score += 10;
                            updateScoreDisplay();
                        }
                        break;
                    }
                }
            }
        }
        Entity* b2 = &bullets_p2[i];
        if (b2->health > 0) {
            for (int j = 0; j < MAX_ENEMIES; j++) {
                Entity* e = &enemies_top[j];
                if (e->health > 0) {
                    if (b2->x < e->x + e->w && b2->x + b2->w > e->x &&
                        b2->y < e->y + e->h && b2->y + b2->h > e->y) {
                        killEntity(b2);
                        bulletsOnScreenP2--;
                        e->health--;
                        if (e->health <= 0) {
                            killEntity(e);
                            enemiesLeft--;
                            score += 10;
                            updateScoreDisplay();
                        }
                        break;
                    }
                }
            }
        }
    }

    // Player bullets vs fly enemies
    if (flyEnemiesSpawned) {
        for (int i = 0; i < MAX_BULLETS; i++) {
            Entity* b1 = &bullets_p1[i];
            if (b1->health > 0) {
                for (int j = 0; j < MAX_FLY_ENEMIES; j++) {
                    Entity* f = &enemies_bottom[j];
                    if (f->health > 0) {
                        if (b1->x < f->x + f->w && b1->x + b1->w > f->x &&
                            b1->y < f->y + f->h && b1->y + b1->h > f->y) {
                            killEntity(b1);
                            bulletsOnScreenP1--;
                            f->health--;
                            if (f->health <= 0) {
                                killEntity(f);
                                score += 15;
                                updateScoreDisplay();
                            }
                            break;
                        }
                    }
                }
            }
            Entity* b2 = &bullets_p2[i];
            if (b2->health > 0) {
                for (int j = 0; j < MAX_FLY_ENEMIES; j++) {
                    Entity* f = &enemies_bottom[j];
                    if (f->health > 0) {
                        if (b2->x < f->x + f->w && b2->x + b2->w > f->x &&
                            b2->y < f->y + f->h && b2->y + b2->h > f->y) {
                            killEntity(b2);
                            bulletsOnScreenP2--;
                            f->health--;
                            if (f->health <= 0) {
                                killEntity(f);
                                score += 15;
                                updateScoreDisplay();
                            }
                            break;
                        }
                    }
                }
            }
        }
    }

    // Enemy bullets vs players
    for (int i = 0; i < MAX_ENEMY_BULLETS; i++) {
        Entity* ebTop = &enemyBulletsTop[i];
        if (ebTop->health > 0) {
            if (ebTop->x < player.x + player.w && ebTop->x + ebTop->w > player.x &&
                ebTop->y < player.y + player.h && ebTop->y + ebTop->h > player.y) {
                killEntity(ebTop);
                enemyBulletsOnScreenTop--;
                // TODO: player hit logic (lives, death, etc)
            }
            if (ebTop->x < player_2.x + player_2.w && ebTop->x + ebTop->w > player_2.x &&
                ebTop->y < player_2.y + player_2.h && ebTop->y + ebTop->h > player_2.y) {
                killEntity(ebTop);
                enemyBulletsOnScreenTop--;
                // TODO: player 2 hit logic
            }
        }
        Entity* ebFly = &enemyBulletsFly[i];
        if (ebFly->health > 0) {
            if (ebFly->x < player.x + player.w && ebFly->x + ebFly->w > player.x &&
                ebFly->y < player.y + player.h && ebFly->y + ebFly->h > player.y) {
                killEntity(ebFly);
                enemyBulletsOnScreenFly--;
                // TODO: player hit logic
            }
            if (ebFly->x < player_2.x + player_2.w && ebFly->x + ebFly->w > player_2.x &&
                ebFly->y < player_2.y + player_2.h && ebFly->y + ebFly->h > player_2.y) {
                killEntity(ebFly);
                enemyBulletsOnScreenFly--;
                // TODO: player 2 hit logic
            }
        }
    }
}

void killEntity(Entity* e) {
    e->health = 0;
    SPR_setVisibility(e->sprite, HIDDEN);
}

void reviveEntity(Entity* e) {
    e->health = 1;
    SPR_setVisibility(e->sprite, VISIBLE);
}

void updateScoreDisplay() {
    sprintf(hud_string, "SCORE: %04d ENEMIES: %02d", score, enemiesLeft);
    VDP_clearTextArea(0, 0, 40, 1);
    VDP_drawText(hud_string, 0, 0);
}

void spawnFlyEnemies() {
    // Example positions (randomized or fixed, but different)
    int spawnX[MAX_FLY_ENEMIES] = {40, 120, 200};
    int spawnY[MAX_FLY_ENEMIES] = {40, 60, 80};

    for (int i = 0; i < MAX_FLY_ENEMIES; i++) {
        if (enemies_bottom[i].sprite == NULL) {
            enemies_bottom[i].sprite = SPR_addSprite(&efly, spawnX[i], spawnY[i], TILE_ATTR(PAL2, 0, TRUE, FALSE));
        }
        enemies_bottom[i].x = spawnX[i];
        enemies_bottom[i].y = spawnY[i];
        enemies_bottom[i].w = 16;
        enemies_bottom[i].h = 16;

        // Start moving right initially, velocity can be positive or negative
        enemies_bottom[i].velx = (i % 2 == 0) ? 1 : -1;
        enemies_bottom[i].vely = 0;

        enemies_bottom[i].health = 1;
        SPR_setAnim(enemies_bottom[i].sprite, 0);
        SPR_setVisibility(enemies_bottom[i].sprite, VISIBLE);
        SPR_setPosition(enemies_bottom[i].sprite, enemies_bottom[i].x, enemies_bottom[i].y);
    }
}

/////////////////CHANGE LOG//////////////////

/*

✅ 1. Two-Player Support
MEGALAGA 7: Only supports Player 1.

BONUS 7: Adds a second player (player_2) with separate controls, bullets (bullets_p2), and collision detection.

Input Handling: Uses JOY_1 and JOY_2 to differentiate between player controls.

🚀 2. New Enemy Type: Fly Enemies
BONUS 7 introduces a second wave of enemies ("fly enemies") that spawn after the top enemies are destroyed.

Delayed Spawn: Controlled by FLY_SPAWN_DELAY (~3 seconds).

Independent Behavior: Have their own movement pattern (horizontal bounce), sprites, and shooting logic.

🔫 3. Separate Enemy Bullet Types
Top Enemies: Fire bullet sprite.

Fly Enemies: Fire bullet1 sprite.

Separate Arrays: enemyBulletsTop and enemyBulletsFly allow distinct management and collision handling.

🧠 4. Smarter Enemy Firing Logic
Top & Fly Enemies now have independent fire cooldowns and fire probabilities.

Uses random firing chances per frame (random() % 20 == 0).

🏆 5. Scoring Improvements
Fly Enemies: Award 15 points on defeat (vs 10 for top enemies).

Score display dynamically updates with:

c
Copy
Edit
sprintf(hud_string, "SCORE: %04d ENEMIES: %02d", score, enemiesLeft);
🪐 6. Improved Sprite & Background Initialization
BONUS 7: More robust background initialization with star patterns.

Players and enemies use palette-specific tiles and flipped sprites for animation.

⚠️ 7. Enhanced Collision Detection
Expanded Cases:

Player 1 bullets vs top & fly enemies

Player 2 bullets vs top & fly enemies

Enemy bullets vs both players

Entity Management: Unified entity logic via killEntity and reviveEntity.

🛠️ 8. Modular Code Organization
Better use of modular functions: positionPlayers(), positionEnemies(), enemyFire(), etc.

Enables easier maintenance and feature extension.

⚙️ 9. Animation and Sprite Handling
Ships animate left/right (ANIM_MOVE) and flip horizontally for direction.

Fly enemies and bullets have controlled visibility and positioning logic.

🔚 Summary Table
Feature	MEGALAGA 7	MEGALAGA BONUS 7
Player Support	1 Player	2 Players
Enemy Types	1 (Top Enemies)	2 (Top + Fly Enemies)
Enemy Bullet Types	1 Shared	Separate per enemy
Score System	Basic	Differentiated scores
Enemy AI	Basic	Advanced w/ cooldowns
Fly Enemy Spawn Logic	❌	✅ Yes, timed
Animation & Flipping	Basic	Direction-aware
Sprite Handling	Simple	Robust, more checks

*/

////////////////////NOTES////////////////////

/*

Code Summary

👥 Players
Player 1 (player) and Player 2 (player_2) are both controlled independently using separate controllers.

Each has:

Position (x, y)

Velocity (velx)

Width & height (16×16)

A Sprite*

A bullet array (bullets_p1, bullets_p2)

Max 3 bullets on screen per player

Movement is constrained to horizontal bounds:


if (p->x < LEFT_EDGE) p->x = LEFT_EDGE;
if (p->x + p->w > RIGHT_EDGE) p->x = RIGHT_EDGE - p->w;
Shooting is handled via button B, with fire rate limited by bullet pool.

👾 Enemies
1. Top Enemies
Static at top of screen, patrol horizontally.

Defined by enemies_top[MAX_ENEMIES].

Each has:

Basic AI: Bounce at screen edges

1 health point

Fires bullets downward randomly

Score: +10 points per kill

2. Fly Enemies (Bonus Feature)
Appear after all top enemies are defeated.

Defined by enemies_bottom[MAX_FLY_ENEMIES].

Spawn after FLY_SPAWN_DELAY (~3 seconds).

Move left/right and fire bullets.

Score: +15 points per kill

💥 Bullets
Player Bullets
3 per player (bullets_p1, bullets_p2)

Move upwards

Use separate sprites (bullet, bullet1)

Destroy on off-screen or on enemy hit

Enemy Bullets
Two separate pools:

enemyBulletsTop: Fired by top enemies

enemyBulletsFly: Fired by fly enemies

Up to 5 bullets in each pool

Move downward and can hit either player

🔫 Shooting System
Players:


shootBullet(p, bullets, bulletCount, -3);
Enemies:

Top and fly enemies fire bullets if:

Their cooldown is 0

Random chance is met

A free bullet slot is available

Cooldowns:


enemyFireCooldownTop = 30;
enemyFireCooldownFly = 30;

🔍 Collision Detection
Handled in handleCollisions():

Player bullets vs enemies (top & fly)

Enemy bullets vs players

Uses simple AABB (axis-aligned bounding box) collision check:


if (b->x < e->x + e->w && b->x + b->w > e->x &&
    b->y < e->y + e->h && b->y + b->h > e->y)
Entities are "killed" by:


killEntity(e); // sets health to 0 and hides sprite

📺 Graphics & Sprites
Uses SGDK’s Sprite system (SPR_* functions).

Player and enemy sprites are animated and flipped based on direction.

Bullets are visually managed via SPR_setPosition() and SPR_setVisibility().

Background:

Scrolling starfield using vertical scroll.

Filled with random star tiles during setup.

💻 Code Structure
Organized modularly:

Initialization:

Load graphics, palettes, tilesets

Init player/enemy structs

Game Loop:

Background scroll

Player/enemy/bullet updates

Collision handling

Sprite updates

Entity Management:

killEntity() and reviveEntity() handle activation/visibility

Scoring:

updateScoreDisplay() updates on-screen HUD

Tracks score and enemies remaining

🧠 AI & Behavior
Top Enemies:

Bounce off screen edges

Fire with a 1-in-20 chance each frame

Fly Enemies:

Spawn after delay once top enemies are gone

Bounce horizontally

Fire bullets with similar random chance

🔧 Technical Details
Constants define screen bounds, enemy caps, etc.

Uses SGDK macros and types:

VDP_* for video layer manipulation

PAL_setPalette(), RGB24_TO_VDPCOLOR() for colors

JOY_* for controller input

Background scrolling resets every 256 pixels

⚙️ Key Definitions

#define MAX_ENEMIES 6
#define MAX_BULLETS 3
#define MAX_ENEMY_BULLETS 5
#define FLY_SPAWN_DELAY 180 // 3 seconds at 60 FPS

*/

/////////EXPERIMENTATION IDEAS///////////////

/*

In the previous bonus we added an additional enemy. Can we give it a different bullet sprite? 

*/

///////////ERROR HANDLING////////////////////
/*

*/